//****** CN131 SDK ******//
//*** SDK Information ***//
// Clock frequency: 80MHz
// Minimum clock frequency : 32MHz
// ADC sampling rate: 250Hz
// ADC resolution：12bit
// Power line frequency filter: 50Hz
// Optional medical standard ECG waveform
// Website: http://www.cyzur.com
// Mail: contactus@cyzur.com
// Release Date: 2023/12/11
// Version: VC.2
// Copyright(C) CyzurTech (Shanghai) Co. Ltd. 2020-
// All rights reserved

#ifndef __CN131_API_H_
#define __CN131_API_H_

#include <string.h>
#include <stdlib.h>
#include <stdint.h>

/****************************************** User board define*********************************************/
// #define DB_AOUT_Pin NRF_SAADC_INPUT_AIN7

/* Define Daughter Board(DB) LOD pin */
// #define DB_LOD1_Pin 3

/* Define Daughter Board(DB) RST pin */
// #define DB_RST_Pin 2

/* Define Daughter Board(DB) SPI pins */
// #define DB_CSB_Pin 30
// #define DB_SCLK_Pin 28
// #define DB_MOSI_Pin 29
// #define DB_MISO_Pin 27

// #define DB_LDO_ENABLE_Pin 4

/****************************************** user board function *********************************************/
/* Wrapper: WritePin Function */
void DB_RST_SET(void);
void DB_RST_RESET(void);
void DB_CSB_SET(void);
void DB_CSB_RESET(void);
void DB_SCLK_SET(void);
void DB_SCLK_RESET(void);
void DB_MOSI_SET(void);
void DB_MOSI_RESET(void);

void DB_LDO_SET(void);
void DB_LDO_RESET(void);

/* Wrapper: ReadPin Function */
uint8_t DB_MISO_HIGH(void);
uint8_t DB_MISO_LOW(void);
uint8_t DB_LOD1_LOW(void);
uint8_t DB_LOD1_HIGH(void);

uint8_t DB_LOD_LOW(void);
uint8_t DB_LOD_HIGH(void);

/*Wrapper: Microsecond delay: us */
void CN1xx_delay_us(uint32_t nus);

/*Wrapper: SPI */
uint8_t CN1xx_SPI_ReadWrite(uint8_t *TxDataSeq, uint8_t *RxDataSeq, uint8_t SeqLen);

/*GPIO Init*/
void GPIO_LDO_Init(void);
void GPIO_LOD_Init(void);
void GPIO_RST_Init(void);
void GPIO_SPI_Init(void);
/****************************************** Cyzur SDK define************************************************/
#define FIFO_Depth 64
#define FIFO_ERROR_LIMITER ((FIFO_Depth * 80) / 100)
#define SAMPLE_RATE 250

// CN131 SPI reg
#define GAIN_Mask 0x0f
#define GAIN_60 0x00
#define GAIN_120 0x01
#define GAIN_180 0x02
#define GAIN_240 0x03
#define GAIN_300 0x04
#define GAIN_360 0x05
#define GAIN_420 0x06
#define GAIN_480 0x07
#define GAIN_540 0x08
#define GAIN_600 0x09
#define GAIN_660 0x0a
#define GAIN_720 0x0b
#define GAIN_780 0x0c
#define GAIN_840 0x0d
#define GAIN_900 0x0e
#define GAIN_960 0x0f
// lp
#define LP_Mask 0x30
#define LP_40 0x00
#define LP_55 0x10
#define LP_150 0x20
#define LP_300 0x30
// hp
#define HP_Mask 0x40
#define HP_4 0x00
#define HP_0p05 0x40
// fr
#define FR_Mask 0x80
#define FR_on 0x80

#define PWR_Mask 0x40
#define PWR_off 0x40

#define STANDBY_Mask 0x20
#define STANDBY_on 0x20

#define OP_RREG 0x20
#define OP_WREG 0x40

#define ADDR_CONFIG 0x00
#define ADDR_CH_SET 0x01

// #define AUTOFR_TRIG_WIN     100  //AutoFR Trigger Window 100ms
// FR
#define AUTOFR_KEEP_LIMIT 50     // ADC_UPDATE_FREQ*AUTOFR_KEEP_TIME/1000
#define AUTOFR_LOW_TRIG_LIMIT 50  // ADC_UPDATE_FREQ*AUTOFR_TRIG_WIN/1000
#define AUTOFR_HIGH_TRIG_LIMIT 50 // ADC_UPDATE_FREQ*AUTOFR_TRIG_WIN/1000

// LEADON
#define LEADON_JUDGE_LIMIT 3  // UNIT:  ADC_UPDATE_FREQ
#define LEADOFF_JUDGE_LIMIT 4 // UNIT:  ADC_UPDATE_FREQ

// Algorithm switch
// #define CN131_ALGORITHM
/****************************************** Cyzur SDK define*********************************************/
/**
 * @brief lod status, indicates whether a lead has fallen off.
 * @note parameter can be selected as below:
 * LEAD_OFF : Leadoff Status
 * LEAD_ON  : Leadon Status
 */
extern volatile uint8_t lod_status;
extern volatile uint16_t heart_Rate;

typedef enum
{
	LEAD_OFF = 0,
	LEAD_ON
} LEAD_STATUS;

typedef enum
{
	Medical_MODE = 0,
	Health_MODE
} CN131_WORKMODE;

typedef enum
{
	ecgOnly = 0,
	realRhythm = 1
} CN131_LOD_On_MODE;

typedef enum
{
	LOD_IT_PROC = 0,
	ADC_IT_PROC,
	GET_VALUE_PROC,
	GET_VALUE_Rdet_PROC
} CN131_PROC;
typedef enum{
    ADC12BIT = 1,
}ADC_RESOLUTION;

typedef enum
{
    AUTO_FR = 0,
    FORCE_FR = 1,
} CN131_FR_MODE;

typedef enum
{
    FR_IDLE = 0,             // 无操作
    FR_TRIGGERED = 1,        // 触发FR
    FR_TIMEOUT_RELEASED = 2, // 超时释放
    FR_ONGOING = 3           // FR保持中
} FR_STATUS;

typedef union
{
    uint8_t bytes[2];   // 原始字节数组，兼容原有接口
    uint16_t half_word; // 整体访问（注意大小端）
    struct
    {
        uint16_t spi_write_read_error : 1; // bit 0
        uint16_t fifo_full_error : 1;      // bit 1
        uint16_t reserved_2 : 1;           // bit 2
        uint16_t reserved_3 : 1;           // bit 3
        uint16_t reserved_4 : 1;           // bit 4
        uint16_t reserved_5 : 1;           // bit 5
        uint16_t reserved_6 : 1;           // bit 6
        uint16_t reserved_7 : 1;           // bit 7
        uint16_t reserved_8 : 1;           // bit 8
        uint16_t reserved_9 : 1;           // bit 9
        uint16_t reserved_10_15 : 6;       // bit 10~15
    } bits;
} ErrorCode;

typedef struct CN1xx_API_Para_t
{
    uint16_t magic;
    uint8_t res[4];
    uint8_t work_mode;
    uint8_t lead_status;
    uint8_t gain;
    uint8_t hp;
    uint8_t lp;
    uint8_t auto_fr_status;
    ErrorCode error_code;
    uint8_t version[2];
    float adc_vol_det;
} CN1xx_API_Para_t;

typedef struct
{
    uint16_t buf[FIFO_Depth];
    uint16_t pos_write;
    uint16_t pos_read;
    uint16_t fifolength;
    uint8_t full;
} FIFO_TypeDef;

/****************************************** Cyzur SDK function *********************************************/
/**
 * @brief
 * @note   Use_For_Standard ： clinical criteria
 * @note       0: Default;
 * @note       1: meet to clinical criteria.
 * @note   ADC_RESOLUTION_12
 * @note      0: ADC RESOLUTION == 10bit;
 * @note      1: Default. ADC RESOLUTION == 12bit.
 * @note   VREF ADC Reference Voltage. Default 3.3
 * @note   VCM Voltage. Default 0.9
 */

void CN131_Parameter_Init(float *lp_parameter, ADC_RESOLUTION ADC_RESOLUTION_12, float VREF, float VVCM);

/**
 * @brief  CN131 initiates procedures for selecting different application domains.
 * @note   The work mode: Medical_MODE or Health_MODE.
 *         retval: 1 for success and 0 for failure.
 * @param  CN131_WORKMODE Enum work_mode
 * @retval uint8_t Bool value
 */
uint8_t CN131_Init(CN131_WORKMODE work_mode);

/**
 * @brief  Use this function to put the CN131 chip start work.
 */
void CN131_Start(void);

/**
 * @brief  Initialize the CN131 chip when the lod_status is lead on.
 */
void CN131_LOD_ON_Init(CN131_LOD_On_MODE init_mode);

/**
 * @brief  CN131 data processing program.
 * @note   There are three modes :LOD_IT_PROC, ADC_IT_PROC, and GET_VALUE_PROC.
 *         Respectively used to acquire ecg signal, used in TIM interrupt, ADC interrupt.
 * @note   retval can be in 3 cases when in LOD_IT_PROC: 0: do not start FR;
 *         1: the start of FR; 2: the end of FR
 * @note   retval can be in 2 cases when in ADC_IT_PROC: 0: the FIFO Push is successful,
 *         1: the FIFO is full and  FIFO Push failed.
 * @note   retval can be in 2 cases when in GET_VALUE_PROC: 0: get_value succeeded;
 *         1: get_value failed because the FIFO state is empty;
 *         2: get_vaue failed because the FIFO state is full,
 * @param  CN131_PROC Enum, int16_t ecg value. int16_t R point delay (relative to the current point)
 * @retval uint8_t Bool value
 */

uint8_t CN131_Data_Proc(CN131_PROC CN131_proc, int16_t *ecg, int16_t *RDelay);

/**
 * @brief  Use this function to put the CN131 chip into standby mode.
 */
uint8_t CN131_Standby(void);

/**
 * @brief  Use this function to stop the CN131 chip.
 * @note   TIM & ADC also stop.
 * @retval uint8_t Bool value
 */
uint8_t CN131_Stop(void);

/**
 * @brief  Use this function to get the size of the cache data inside the SDK.
 * @note   ADC' value push this cache.
 * @retval uint16_t the cache data
 */
uint16_t FIFO_Size(void);

void CN131_LOD_Detect(void);

uint32_t Get_CN1xx_parameter(CN1xx_API_Para_t *parameter);
#endif /* CN131_API_H_ */
